/* Custom javascript 详细介绍请查看官方文档
 * @param {Object} params
 * @property {Array} params.axes  chart options中的轴配置，与拖入的轴对应
 * @property {Object} params.data  接口返回的数据，与轴对应
 * @property {Array} params.data.data 接口返回的原始数据
 * @property {Array} params.data.schema 这里包含了初步加工过的数据，其中的dataUniq是去重并按照配置排好序的
 * @property {number} params.width  容器的宽度
 * @property {number} params.height 容器的高度
 * @property {Function} params.getD3 获取d3工具库
 * @property {Function} params.getEcharts 获取echarts，当前版本是4.9，用于写扩展
 * @property {Function} params.getEchartsItem 获取echarts实例
 * @property {Function} params.setInnerHTML 设置容器的innerHTML
 * @property {Function} params.bindClickListener 绑定页面点击事件
 * @property {Function} params.onHSClickHandler 触发衡石交互
 * @property {Function} params.getContext 获取canvas的context
 * @property {Object=} params.currentHighlight 当前高亮项
 * @property {Array<clickEventData>=} params.currentHighlight.clickEventData 当前高亮项对应的数据
 */
console.log(params);

var axes = params.axes;
if (!axes || axes.length !== 5 || axes[2].axisName !== 'group' || axes[3].axisName !== 'metric') {
  // 如果不是一个维度一个度量，提示
  params.setInnerHTML('<div style="text-align:center;line-height:' + params.height + 'px;">'
  +    '<span>此图表只支持3个维度和2个度量的配置，请在配置区编辑</span>'
  +  '</div>'
  );
  return;
}

const THREE = params.getThree();
const TWEEN = params.getTween();

// 定义 TrackballControls
const {
  EventDispatcher,
  MOUSE,
  Quaternion,
  Vector2,
  Vector3,
    Matrix4,
  Object3D
} = THREE;
const _changeEvent = { type: 'change' };
const _startEvent = { type: 'start' };
const _endEvent = { type: 'end' };

class TrackballControls extends EventDispatcher {

  constructor( object, domElement ) {

    super();

    if ( domElement === undefined ) console.warn( 'THREE.TrackballControls: The second parameter "domElement" is now mandatory.' );
    if ( domElement === document ) console.error( 'THREE.TrackballControls: "document" should not be used as the target "domElement". Please use "renderer.domElement" instead.' );

    const scope = this;
    const STATE = { NONE: - 1, ROTATE: 0, ZOOM: 1, PAN: 2, TOUCH_ROTATE: 3, TOUCH_ZOOM_PAN: 4 };

    this.object = object;
    this.domElement = domElement;
    this.domElement.style.touchAction = 'none'; // disable touch scroll

    // API

    this.enabled = true;

    this.screen = { left: 0, top: 0, width: 0, height: 0 };

    this.rotateSpeed = 1.0;
    this.zoomSpeed = 1.2;
    this.panSpeed = 0.3;

    this.noRotate = false;
    this.noZoom = false;
    this.noPan = false;

    this.staticMoving = false;
    this.dynamicDampingFactor = 0.2;

    this.minDistance = 0;
    this.maxDistance = Infinity;

    this.keys = [ 'KeyA' /*A*/, 'KeyS' /*S*/, 'KeyD' /*D*/ ];

    this.mouseButtons = { LEFT: MOUSE.ROTATE, MIDDLE: MOUSE.DOLLY, RIGHT: MOUSE.PAN };

    // internals

    this.target = new Vector3();

    const EPS = 0.000001;

    const lastPosition = new Vector3();
    let lastZoom = 1;

    let _state = STATE.NONE,
      _keyState = STATE.NONE,

      _touchZoomDistanceStart = 0,
      _touchZoomDistanceEnd = 0,

      _lastAngle = 0;

    const _eye = new Vector3(),

      _movePrev = new Vector2(),
      _moveCurr = new Vector2(),

      _lastAxis = new Vector3(),

      _zoomStart = new Vector2(),
      _zoomEnd = new Vector2(),

      _panStart = new Vector2(),
      _panEnd = new Vector2(),

      _pointers = [],
      _pointerPositions = {};

    // for reset

    this.target0 = this.target.clone();
    this.position0 = this.object.position.clone();
    this.up0 = this.object.up.clone();
    this.zoom0 = this.object.zoom;

    // methods

    this.handleResize = function () {

      const box = scope.domElement.getBoundingClientRect();
      // adjustments come from similar code in the jquery offset() function
      const d = scope.domElement.ownerDocument.documentElement;
      scope.screen.left = box.left + 0 - d.clientLeft;
      scope.screen.top = box.top + 0 - d.clientTop;
      scope.screen.width = box.width;
      scope.screen.height = box.height;

    };

    const getMouseOnScreen = ( function () {

      const vector = new Vector2();

      return function getMouseOnScreen( pageX, pageY ) {

        vector.set(
          ( pageX - scope.screen.left ) / scope.screen.width,
          ( pageY - scope.screen.top ) / scope.screen.height
        );

        return vector;

      };

    }() );

    const getMouseOnCircle = ( function () {

      const vector = new Vector2();

      return function getMouseOnCircle( pageX, pageY ) {

        vector.set(
          ( ( pageX - scope.screen.width * 0.5 - scope.screen.left ) / ( scope.screen.width * 0.5 ) ),
          ( ( scope.screen.height + 2 * ( scope.screen.top - pageY ) ) / scope.screen.width ) // screen.width intentional
        );

        return vector;

      };

    }() );

    this.rotateCamera = ( function () {

      const axis = new Vector3(),
        quaternion = new Quaternion(),
        eyeDirection = new Vector3(),
        objectUpDirection = new Vector3(),
        objectSidewaysDirection = new Vector3(),
        moveDirection = new Vector3();

      return function rotateCamera() {

        moveDirection.set( _moveCurr.x - _movePrev.x, _moveCurr.y - _movePrev.y, 0 );
        let angle = moveDirection.length();

        if ( angle ) {

          _eye.copy( scope.object.position ).sub( scope.target );

          eyeDirection.copy( _eye ).normalize();
          objectUpDirection.copy( scope.object.up ).normalize();
          objectSidewaysDirection.crossVectors( objectUpDirection, eyeDirection ).normalize();

          objectUpDirection.setLength( _moveCurr.y - _movePrev.y );
          objectSidewaysDirection.setLength( _moveCurr.x - _movePrev.x );

          moveDirection.copy( objectUpDirection.add( objectSidewaysDirection ) );

          axis.crossVectors( moveDirection, _eye ).normalize();

          angle *= scope.rotateSpeed;
          quaternion.setFromAxisAngle( axis, angle );

          _eye.applyQuaternion( quaternion );
          scope.object.up.applyQuaternion( quaternion );

          _lastAxis.copy( axis );
          _lastAngle = angle;

        } else if ( ! scope.staticMoving && _lastAngle ) {

          _lastAngle *= Math.sqrt( 1.0 - scope.dynamicDampingFactor );
          _eye.copy( scope.object.position ).sub( scope.target );
          quaternion.setFromAxisAngle( _lastAxis, _lastAngle );
          _eye.applyQuaternion( quaternion );
          scope.object.up.applyQuaternion( quaternion );

        }

        _movePrev.copy( _moveCurr );

      };

    }() );


    this.zoomCamera = function () {

      let factor;

      if ( _state === STATE.TOUCH_ZOOM_PAN ) {

        factor = _touchZoomDistanceStart / _touchZoomDistanceEnd;
        _touchZoomDistanceStart = _touchZoomDistanceEnd;

        if ( scope.object.isPerspectiveCamera ) {

          _eye.multiplyScalar( factor );

        } else if ( scope.object.isOrthographicCamera ) {

          scope.object.zoom *= factor;
          scope.object.updateProjectionMatrix();

        } else {

          console.warn( 'THREE.TrackballControls: Unsupported camera type' );

        }

      } else {

        factor = 1.0 + ( _zoomEnd.y - _zoomStart.y ) * scope.zoomSpeed;

        if ( factor !== 1.0 && factor > 0.0 ) {

          if ( scope.object.isPerspectiveCamera ) {

            _eye.multiplyScalar( factor );

          } else if ( scope.object.isOrthographicCamera ) {

            scope.object.zoom /= factor;
            scope.object.updateProjectionMatrix();

          } else {

            console.warn( 'THREE.TrackballControls: Unsupported camera type' );

          }

        }

        if ( scope.staticMoving ) {

          _zoomStart.copy( _zoomEnd );

        } else {

          _zoomStart.y += ( _zoomEnd.y - _zoomStart.y ) * this.dynamicDampingFactor;

        }

      }

    };

    this.panCamera = ( function () {

      const mouseChange = new Vector2(),
        objectUp = new Vector3(),
        pan = new Vector3();

      return function panCamera() {

        mouseChange.copy( _panEnd ).sub( _panStart );

        if ( mouseChange.lengthSq() ) {

          if ( scope.object.isOrthographicCamera ) {

            const scale_x = ( scope.object.right - scope.object.left ) / scope.object.zoom / scope.domElement.clientWidth;
            const scale_y = ( scope.object.top - scope.object.bottom ) / scope.object.zoom / scope.domElement.clientWidth;

            mouseChange.x *= scale_x;
            mouseChange.y *= scale_y;

          }

          mouseChange.multiplyScalar( _eye.length() * scope.panSpeed );

          pan.copy( _eye ).cross( scope.object.up ).setLength( mouseChange.x );
          pan.add( objectUp.copy( scope.object.up ).setLength( mouseChange.y ) );

          scope.object.position.add( pan );
          scope.target.add( pan );

          if ( scope.staticMoving ) {

            _panStart.copy( _panEnd );

          } else {

            _panStart.add( mouseChange.subVectors( _panEnd, _panStart ).multiplyScalar( scope.dynamicDampingFactor ) );

          }

        }

      };

    }() );

    this.checkDistances = function () {

      if ( ! scope.noZoom || ! scope.noPan ) {

        if ( _eye.lengthSq() > scope.maxDistance * scope.maxDistance ) {

          scope.object.position.addVectors( scope.target, _eye.setLength( scope.maxDistance ) );
          _zoomStart.copy( _zoomEnd );

        }

        if ( _eye.lengthSq() < scope.minDistance * scope.minDistance ) {

          scope.object.position.addVectors( scope.target, _eye.setLength( scope.minDistance ) );
          _zoomStart.copy( _zoomEnd );

        }

      }

    };

    this.update = function () {

      _eye.subVectors( scope.object.position, scope.target );

      if ( ! scope.noRotate ) {

        scope.rotateCamera();

      }

      if ( ! scope.noZoom ) {

        scope.zoomCamera();

      }

      if ( ! scope.noPan ) {

        scope.panCamera();

      }

      scope.object.position.addVectors( scope.target, _eye );

      if ( scope.object.isPerspectiveCamera ) {

        scope.checkDistances();

        scope.object.lookAt( scope.target );

        if ( lastPosition.distanceToSquared( scope.object.position ) > EPS ) {

          scope.dispatchEvent( _changeEvent );

          lastPosition.copy( scope.object.position );

        }

      } else if ( scope.object.isOrthographicCamera ) {

        scope.object.lookAt( scope.target );

        if ( lastPosition.distanceToSquared( scope.object.position ) > EPS || lastZoom !== scope.object.zoom ) {

          scope.dispatchEvent( _changeEvent );

          lastPosition.copy( scope.object.position );
          lastZoom = scope.object.zoom;

        }

      } else {

        console.warn( 'THREE.TrackballControls: Unsupported camera type' );

      }

    };

    this.reset = function () {

      _state = STATE.NONE;
      _keyState = STATE.NONE;

      scope.target.copy( scope.target0 );
      scope.object.position.copy( scope.position0 );
      scope.object.up.copy( scope.up0 );
      scope.object.zoom = scope.zoom0;

      scope.object.updateProjectionMatrix();

      _eye.subVectors( scope.object.position, scope.target );

      scope.object.lookAt( scope.target );

      scope.dispatchEvent( _changeEvent );

      lastPosition.copy( scope.object.position );
      lastZoom = scope.object.zoom;

    };

    // listeners

    function onPointerDown( event ) {

      if ( scope.enabled === false ) return;

      if ( _pointers.length === 0 ) {

        scope.domElement.setPointerCapture( event.pointerId );

        scope.domElement.addEventListener( 'pointermove', onPointerMove );
        scope.domElement.addEventListener( 'pointerup', onPointerUp );

      }

      //

      addPointer( event );

      if ( event.pointerType === 'touch' ) {

        onTouchStart( event );

      } else {

        onMouseDown( event );

      }

    }

    function onPointerMove( event ) {

      if ( scope.enabled === false ) return;

      if ( event.pointerType === 'touch' ) {

        onTouchMove( event );

      } else {

        onMouseMove( event );

      }

    }

    function onPointerUp( event ) {

      if ( scope.enabled === false ) return;

      if ( event.pointerType === 'touch' ) {

        onTouchEnd( event );

      } else {

        onMouseUp();

      }

      //

      removePointer( event );

      if ( _pointers.length === 0 ) {

        scope.domElement.releasePointerCapture( event.pointerId );

        scope.domElement.removeEventListener( 'pointermove', onPointerMove );
        scope.domElement.removeEventListener( 'pointerup', onPointerUp );

      }


    }

    function onPointerCancel( event ) {

      removePointer( event );

    }

    function onMouseDown( event ) {

      if ( _state === STATE.NONE ) {

        switch ( event.button ) {

          case scope.mouseButtons.LEFT:
            _state = STATE.ROTATE;
            break;

          case scope.mouseButtons.MIDDLE:
            _state = STATE.ZOOM;
            break;

          case scope.mouseButtons.RIGHT:
            _state = STATE.PAN;
            break;

          default:
            _state = STATE.NONE;

        }

      }

      const state = ( _keyState !== STATE.NONE ) ? _keyState : _state;

      if ( state === STATE.ROTATE && ! scope.noRotate ) {

        _moveCurr.copy( getMouseOnCircle( event.pageX, event.pageY ) );
        _movePrev.copy( _moveCurr );

      } else if ( state === STATE.ZOOM && ! scope.noZoom ) {

        _zoomStart.copy( getMouseOnScreen( event.pageX, event.pageY ) );
        _zoomEnd.copy( _zoomStart );

      } else if ( state === STATE.PAN && ! scope.noPan ) {

        _panStart.copy( getMouseOnScreen( event.pageX, event.pageY ) );
        _panEnd.copy( _panStart );

      }

      scope.dispatchEvent( _startEvent );

    }

    function onMouseMove( event ) {

      const state = ( _keyState !== STATE.NONE ) ? _keyState : _state;

      if ( state === STATE.ROTATE && ! scope.noRotate ) {

        _movePrev.copy( _moveCurr );
        _moveCurr.copy( getMouseOnCircle( event.pageX, event.pageY ) );

      } else if ( state === STATE.ZOOM && ! scope.noZoom ) {

        _zoomEnd.copy( getMouseOnScreen( event.pageX, event.pageY ) );

      } else if ( state === STATE.PAN && ! scope.noPan ) {

        _panEnd.copy( getMouseOnScreen( event.pageX, event.pageY ) );

      }

    }

    function onMouseUp() {

      _state = STATE.NONE;

      scope.dispatchEvent( _endEvent );

    }

    function onMouseWheel( event ) {

      if ( scope.enabled === false ) return;

      if ( scope.noZoom === true ) return;

      event.preventDefault();

      switch ( event.deltaMode ) {

        case 2:
          // Zoom in pages
          _zoomStart.y -= event.deltaY * 0.025;
          break;

        case 1:
          // Zoom in lines
          _zoomStart.y -= event.deltaY * 0.01;
          break;

        default:
          // undefined, 0, assume pixels
          _zoomStart.y -= event.deltaY * 0.00025;
          break;

      }

      scope.dispatchEvent( _startEvent );
      scope.dispatchEvent( _endEvent );

    }

    function onTouchStart( event ) {

      trackPointer( event );

      switch ( _pointers.length ) {

        case 1:
          _state = STATE.TOUCH_ROTATE;
          _moveCurr.copy( getMouseOnCircle( _pointers[ 0 ].pageX, _pointers[ 0 ].pageY ) );
          _movePrev.copy( _moveCurr );
          break;

        default: // 2 or more
          _state = STATE.TOUCH_ZOOM_PAN;
          const dx = _pointers[ 0 ].pageX - _pointers[ 1 ].pageX;
          const dy = _pointers[ 0 ].pageY - _pointers[ 1 ].pageY;
          _touchZoomDistanceEnd = _touchZoomDistanceStart = Math.sqrt( dx * dx + dy * dy );

          const x = ( _pointers[ 0 ].pageX + _pointers[ 1 ].pageX ) / 2;
          const y = ( _pointers[ 0 ].pageY + _pointers[ 1 ].pageY ) / 2;
          _panStart.copy( getMouseOnScreen( x, y ) );
          _panEnd.copy( _panStart );
          break;

      }

      scope.dispatchEvent( _startEvent );

    }

    function onTouchMove( event ) {

      trackPointer( event );

      switch ( _pointers.length ) {

        case 1:
          _movePrev.copy( _moveCurr );
          _moveCurr.copy( getMouseOnCircle( event.pageX, event.pageY ) );
          break;

        default: // 2 or more

          const position = getSecondPointerPosition( event );

          const dx = event.pageX - position.x;
          const dy = event.pageY - position.y;
          _touchZoomDistanceEnd = Math.sqrt( dx * dx + dy * dy );

          const x = ( event.pageX + position.x ) / 2;
          const y = ( event.pageY + position.y ) / 2;
          _panEnd.copy( getMouseOnScreen( x, y ) );
          break;

      }

    }

    function onTouchEnd( event ) {

      switch ( _pointers.length ) {

        case 0:
          _state = STATE.NONE;
          break;

        case 1:
          _state = STATE.TOUCH_ROTATE;
          _moveCurr.copy( getMouseOnCircle( event.pageX, event.pageY ) );
          _movePrev.copy( _moveCurr );
          break;

        case 2:
          _state = STATE.TOUCH_ZOOM_PAN;
          _moveCurr.copy( getMouseOnCircle( event.pageX - _movePrev.pageX, event.pageY - _movePrev.pageY ) );
          _movePrev.copy( _moveCurr );
          break;

      }

      scope.dispatchEvent( _endEvent );

    }

    function contextmenu( event ) {

      if ( scope.enabled === false ) return;

      event.preventDefault();

    }

    function addPointer( event ) {

      _pointers.push( event );

    }

    function removePointer( event ) {

      delete _pointerPositions[ event.pointerId ];

      for ( let i = 0; i < _pointers.length; i ++ ) {

        if ( _pointers[ i ].pointerId == event.pointerId ) {

          _pointers.splice( i, 1 );
          return;

        }

      }

    }

    function trackPointer( event ) {

      let position = _pointerPositions[ event.pointerId ];

      if ( position === undefined ) {

        position = new Vector2();
        _pointerPositions[ event.pointerId ] = position;

      }

      position.set( event.pageX, event.pageY );

    }

    function getSecondPointerPosition( event ) {

      const pointer = ( event.pointerId === _pointers[ 0 ].pointerId ) ? _pointers[ 1 ] : _pointers[ 0 ];

      return _pointerPositions[ pointer.pointerId ];

    }

    this.dispose = function () {

      scope.domElement.removeEventListener( 'contextmenu', contextmenu );

      scope.domElement.removeEventListener( 'pointerdown', onPointerDown );
      scope.domElement.removeEventListener( 'pointercancel', onPointerCancel );
      scope.domElement.removeEventListener( 'wheel', onMouseWheel );

      scope.domElement.removeEventListener( 'pointermove', onPointerMove );
      scope.domElement.removeEventListener( 'pointerup', onPointerUp );
    };

    this.domElement.addEventListener( 'contextmenu', contextmenu );

    this.domElement.addEventListener( 'pointerdown', onPointerDown );
    this.domElement.addEventListener( 'pointerup', onPointerUp );

    this.domElement.addEventListener( 'pointercancel', onPointerCancel );
    this.domElement.addEventListener( 'wheel', onMouseWheel, { passive: false } );

    this.handleResize();

    // force an update at start
    this.update();

  }

}

// 定义CSS3DRender
const _position = new Vector3();
const _quaternion = new Quaternion();
const _scale = new Vector3();

class CSS3DObject extends Object3D {

  constructor( element ) {

    super();

    this.element = element || params.createElement( 'div' );
    this.element.style.position = 'absolute';
    this.element.style.pointerEvents = 'auto';
    this.element.style.userSelect = 'none';

    this.element.setAttribute( 'draggable', false );

    this.addEventListener( 'removed', function () {

      this.traverse( function ( object ) {

        if ( object.element instanceof Element && params.getParentDOM(object.element) !== null ) {

          params.getParentDOM(object.element).removeChild( object.element );

        }

      } );

    } );

  }

  copy( source, recursive ) {

    super.copy( source, recursive );

    this.element = source.element.cloneNode( true );

    return this;

  }

}

CSS3DObject.prototype.isCSS3DObject = true;
//

const _matrix = new Matrix4();
const _matrix2 = new Matrix4();

class CSS3DRenderer {

  constructor() {

    const _this = this;

    let _width, _height;
    let _widthHalf, _heightHalf;

    const cache = {
      camera: { fov: 0, style: '' },
      objects: new WeakMap()
    };

    const domElement = params.createElement( 'div' );
    domElement.style.overflow = 'hidden';

    this.domElement = domElement;

    const cameraElement = params.createElement( 'div' );

    cameraElement.style.transformStyle = 'preserve-3d';
    // cameraElement.style.pointerEvents = 'none';

    domElement.appendChild( cameraElement );

    //获取事件操作对象
    function getSelsectOBj(raycaster, e) {
      //将html坐标系转化为webgl坐标系，并确定鼠标点击位置
      const pointer = {
        x: e.offsetX / renderer.domElement.clientWidth*2-1,
        y: -(e.offsetY / renderer.domElement.clientHeight*2)+1,
        button: e.button,
      };
      //以camera为z坐标，确定所点击物体的3D空间位置
      raycaster.setFromCamera(pointer,camera);
      //确定所点击位置上的物体数量
      let intersects = raycaster.intersectObjects(scene.children, true);
      console.log(raycaster, pointer, e, scene, camera);
      return intersects;
    }

    this.getSize = function () {

      return {
        width: _width,
        height: _height
      };

    };

    this.render = function ( scene, camera ) {

      const fov = camera.projectionMatrix.elements[ 5 ] * _heightHalf;

      if ( cache.camera.fov !== fov ) {

        domElement.style.perspective = camera.isPerspectiveCamera ? fov + 'px' : '';
        cache.camera.fov = fov;

      }

      if ( scene.autoUpdate === true ) scene.updateMatrixWorld();
      if ( camera.parent === null ) camera.updateMatrixWorld();

      let tx, ty;

      if ( camera.isOrthographicCamera ) {

        tx = - ( camera.right + camera.left ) / 2;
        ty = ( camera.top + camera.bottom ) / 2;

      }

      const cameraCSSMatrix = camera.isOrthographicCamera ?
        'scale(' + fov + ')' + 'translate(' + epsilon( tx ) + 'px,' + epsilon( ty ) + 'px)' + getCameraCSSMatrix( camera.matrixWorldInverse ) :
        'translateZ(' + fov + 'px)' + getCameraCSSMatrix( camera.matrixWorldInverse );

      const style = cameraCSSMatrix +
        'translate(' + _widthHalf + 'px,' + _heightHalf + 'px)';

      if ( cache.camera.style !== style ) {

        cameraElement.style.transform = style;

        cache.camera.style = style;

      }

      renderObject( scene, scene, camera, cameraCSSMatrix );

    };

    this.setSize = function ( width, height ) {

      _width = width;
      _height = height;
      _widthHalf = _width / 2;
      _heightHalf = _height / 2;

      domElement.style.width = width + 'px';
      domElement.style.height = height + 'px';

      cameraElement.style.width = width + 'px';
      cameraElement.style.height = height + 'px';

    };

    function epsilon( value ) {

      return Math.abs( value ) < 1e-10 ? 0 : value;

    }

    function getCameraCSSMatrix( matrix ) {

      const elements = matrix.elements;

      return 'matrix3d(' +
        epsilon( elements[ 0 ] ) + ',' +
        epsilon( - elements[ 1 ] ) + ',' +
        epsilon( elements[ 2 ] ) + ',' +
        epsilon( elements[ 3 ] ) + ',' +
        epsilon( elements[ 4 ] ) + ',' +
        epsilon( - elements[ 5 ] ) + ',' +
        epsilon( elements[ 6 ] ) + ',' +
        epsilon( elements[ 7 ] ) + ',' +
        epsilon( elements[ 8 ] ) + ',' +
        epsilon( - elements[ 9 ] ) + ',' +
        epsilon( elements[ 10 ] ) + ',' +
        epsilon( elements[ 11 ] ) + ',' +
        epsilon( elements[ 12 ] ) + ',' +
        epsilon( - elements[ 13 ] ) + ',' +
        epsilon( elements[ 14 ] ) + ',' +
        epsilon( elements[ 15 ] ) +
      ')';

    }

    function getObjectCSSMatrix( matrix ) {

      const elements = matrix.elements;
      const matrix3d = 'matrix3d(' +
        epsilon( elements[ 0 ] ) + ',' +
        epsilon( elements[ 1 ] ) + ',' +
        epsilon( elements[ 2 ] ) + ',' +
        epsilon( elements[ 3 ] ) + ',' +
        epsilon( - elements[ 4 ] ) + ',' +
        epsilon( - elements[ 5 ] ) + ',' +
        epsilon( - elements[ 6 ] ) + ',' +
        epsilon( - elements[ 7 ] ) + ',' +
        epsilon( elements[ 8 ] ) + ',' +
        epsilon( elements[ 9 ] ) + ',' +
        epsilon( elements[ 10 ] ) + ',' +
        epsilon( elements[ 11 ] ) + ',' +
        epsilon( elements[ 12 ] ) + ',' +
        epsilon( elements[ 13 ] ) + ',' +
        epsilon( elements[ 14 ] ) + ',' +
        epsilon( elements[ 15 ] ) +
      ')';

      return 'translate(-50%,-50%)' + matrix3d;

    }

    function renderObject( object, scene, camera, cameraCSSMatrix ) {

      if ( object.isCSS3DObject ) {

        object.onBeforeRender( _this, scene, camera );

        let style;

        if ( object.isCSS3DSprite ) {

          // http://swiftcoder.wordpress.com/2008/11/25/constructing-a-billboard-matrix/

          _matrix.copy( camera.matrixWorldInverse );
          _matrix.transpose();

          if ( object.rotation2D !== 0 ) _matrix.multiply( _matrix2.makeRotationZ( object.rotation2D ) );

          object.matrixWorld.decompose( _position, _quaternion, _scale );
          _matrix.setPosition( _position );
          _matrix.scale( _scale );

          _matrix.elements[ 3 ] = 0;
          _matrix.elements[ 7 ] = 0;
          _matrix.elements[ 11 ] = 0;
          _matrix.elements[ 15 ] = 1;

          style = getObjectCSSMatrix( _matrix );

        } else {

          style = getObjectCSSMatrix( object.matrixWorld );

        }

        const element = object.element;
        const cachedObject = cache.objects.get( object );

        if ( cachedObject === undefined || cachedObject.style !== style ) {

          element.style.transform = style;

          const objectData = { style: style };
          cache.objects.set( object, objectData );

        }

        element.style.display = object.visible ? '' : 'none';

        if ( params.getParentDOM(element) !== cameraElement ) {

          cameraElement.appendChild( element );

        }

        object.onAfterRender( _this, scene, camera );

      }

      for ( let i = 0, l = object.children.length; i < l; i ++ ) {

        renderObject( object.children[ i ], scene, camera, cameraCSSMatrix );

      }

    }

  }

}

const table = params.data.data;
let camera, scene, renderer;
let controls;

const objects = [];
const targets = { table: [], sphere: [], helix: [], grid: [] };

if (!params.isHightlightChange) {
  init();
  animate();
}

let _clickTimeout;

function init() {

  params.setInnerHTML('');
  camera = new THREE.PerspectiveCamera( 40, params.width / params.height, 1, 10000 );

  camera.position.z = 3000;

  scene = new THREE.Scene();

  // table

  for ( let i = 0; i < table.length; i += 1 ) {
    const element = params.createElement( 'div' );
    element.className = 'element';
    element.style.backgroundColor = 'rgba(0,127,127,' + ( Math.random() * 0.5 + 0.25 ) + ')';
    element.style.width = '120px';
    element.style.height = '160px';
    element.style.boxShadow = '0px 0px 12px rgba(0,255,255,0.5)';
    element.style.border = '1px solid rgba(127,255,255,0.25)';
    element.style.textAlign = 'center';
    element.style.cursor = 'pointer';
    element.onpointerdown = function () {
      _clickTimeout = setTimeout(function () {
        params.onHSClickHandler([{
          data: table[ i ][0],
          path: 0
        }]);
      }, 100);
    };
    element.onpointermove = function () {
      clearTimeout(_clickTimeout);
    }

    const number = params.createElement( 'div' );
    number.className = 'number';
    number.textContent = ( i / 5 ) + 1;
    number.style.position = 'absolute';
    number.style.top = '20px';
    number.style.right = '20px';
    number.style.fontSize = '12px';
    number.style.color = 'rgba(127,255,255,0.75)';
    element.appendChild( number );

    const symbol = params.createElement( 'div' );
    symbol.className = 'symbol';
    symbol.textContent = table[ i ][0];
    symbol.style.position = 'absolute';
    symbol.style.top = '40px';
    symbol.style.left = 0;
    symbol.style.right = 0;
    symbol.style.fontSize = '60px';
    symbol.style.fontWeight = 'bold';
    symbol.style.color = 'rgba(255,255,255,0.75)';
    symbol.style.textShadow = '0 0 10px rgba(0,255,255,0.95)';
    element.appendChild( symbol );

    const details = params.createElement( 'div' );
    details.className = 'details';
    details.innerHTML = table[i][1] + '<br>' + table[i][2];
    details.style.position = 'absolute';
    details.style.bottom = '15px';
    details.style.left = 0;
    details.style.right = 0;
    details.style.fontSize = '12px';
    details.style.color = 'rgba(127,255,255,0.75)';
    element.appendChild( details );

    const objectCSS = new CSS3DObject( element );
    objectCSS.position.x = Math.random() * 4000 - 2000;
    objectCSS.position.y = Math.random() * 4000 - 2000;
    objectCSS.position.z = Math.random() * 4000 - 2000;
    scene.add( objectCSS );

    objects.push( objectCSS );

    //

    const object = new THREE.Object3D();
    object.position.x = ( table[i][ 3 ] * 140 ) - 1330;
    object.position.y = - ( table[i][ 4 ] * 180 ) + 990;

    targets.table.push( object );

  }
  // sphere

  const vector = new THREE.Vector3();

  for ( let i = 0, l = objects.length; i < l; i ++ ) {

    const phi = Math.acos( - 1 + ( 2 * i ) / l );
    const theta = Math.sqrt( l * Math.PI ) * phi;

    const object = new THREE.Object3D();

    object.position.setFromSphericalCoords( 800, phi, theta );

    vector.copy( object.position ).multiplyScalar( 2 );

    object.lookAt( vector );

    targets.sphere.push( object );

  }

  // helix

  for ( let i = 0, l = objects.length; i < l; i ++ ) {

    const theta = i * 0.175 + Math.PI;
    const y = - ( i * 8 ) + 450;

    const object = new THREE.Object3D();

    object.position.setFromCylindricalCoords( 900, theta, y );

    vector.x = object.position.x * 2;
    vector.y = object.position.y;
    vector.z = object.position.z * 2;

    object.lookAt( vector );

    targets.helix.push( object );

  }

  // grid

  for ( let i = 0; i < objects.length; i ++ ) {

    const object = new THREE.Object3D();

    object.position.x = ( ( i % 5 ) * 400 ) - 800;
    object.position.y = ( - ( Math.floor( i / 5 ) % 5 ) * 400 ) + 800;
    object.position.z = ( Math.floor( i / 25 ) ) * 1000 - 2000;

    targets.grid.push( object );

  }

  //

  renderer = new CSS3DRenderer();
  renderer.setSize( params.width, params.height );
  params.appendChild(renderer.domElement);
  renderer.domElement.style.backgroundColor = '#000';

  //

  controls = new TrackballControls( camera, renderer.domElement );
  controls.minDistance = 500;
  controls.maxDistance = 6000;
  controls.addEventListener( 'change', render );

  const buttonTable = params.createElement( 'span' );
  buttonTable.style.color = '#fff';
  buttonTable.style.padding = '20px';
  buttonTable.innerText = 'Table';

  buttonTable.addEventListener( 'click', function () {

    transform( targets.table, 2000 );

  } );

  const buttonSphere = params.createElement( 'span' );
  buttonSphere.addEventListener( 'click', function () {

    transform( targets.sphere, 2000 );

  } );
  buttonSphere.style.color = '#fff';
  buttonSphere.style.padding = '20px';
  buttonSphere.innerText = 'Sphere';


  const buttonHelix = params.createElement( 'span' );
  buttonHelix.addEventListener( 'click', function () {

    transform( targets.helix, 2000 );

  } );
  buttonHelix.style.color = '#fff';
  buttonHelix.style.padding = '20px';
  buttonHelix.innerText = 'Helix';


  const buttonGrid = params.createElement( 'span' );
  buttonGrid.addEventListener( 'click', function () {

    transform( targets.grid, 2000 );

  } );
  buttonGrid.style.color = '#fff';
  buttonGrid.style.padding = '20px';
  buttonGrid.innerText = 'Grid';


  const buttonContainer = params.createElement('div');
  buttonContainer.style.position = 'absolute';
  buttonContainer.style.top = 0;
  buttonContainer.style.left = 0;
  buttonContainer.style.right = 0;
  buttonContainer.style.textAlign = 'center';
  buttonContainer.appendChild(buttonTable);
  buttonContainer.appendChild(buttonSphere);
  buttonContainer.appendChild(buttonHelix);
  buttonContainer.appendChild(buttonGrid);
  params.appendChild(buttonContainer);


  transform( targets.table, 2000 );

  //

}

function transform( targets, duration ) {

  TWEEN.removeAll();

  for ( let i = 0; i < objects.length; i ++ ) {

    const object = objects[ i ];
    const target = targets[ i ];

    new TWEEN.Tween( object.position )
      .to( { x: target.position.x, y: target.position.y, z: target.position.z }, Math.random() * duration + duration )
      .easing( TWEEN.Easing.Exponential.InOut )
      .start();

    new TWEEN.Tween( object.rotation )
      .to( { x: target.rotation.x, y: target.rotation.y, z: target.rotation.z }, Math.random() * duration + duration )
      .easing( TWEEN.Easing.Exponential.InOut )
      .start();

  }

  new TWEEN.Tween( this )
    .to( {}, duration * 2 )
    .onUpdate( render )
    .start();

}

function animate() {

  requestAnimationFrame( animate );

  TWEEN.update();

  controls.update();

}

function render() {

  renderer.render( scene, camera );

}

/**
 * @typedef {Object} clickOptions
 * @property {string} className  点击事件的当前元素的className
 * @property {string} id 点击事件的当前元素的id
 * @property {number} x 当前鼠标在容器中的横向相对位置
 * @property {number} y 当前鼠标在容器中的纵向相对位置
 * @property {string} data 当前元素的data属性
 */

/**
 * @type {Function} params.bindClickListener 绑定页面点击事件
 * @param {Function} clickHandler 点击页面的处理函数，接收一个参数clickOptions
 */

/**
 * @typedef ClickEventData
 * @param {string} data 维度值
 * @param {string | number} path 维度值对应的轴位置，例：0（第一个轴）； '1.1' (第二个轴的第二个子轴)
 */

/**
 * @type {Function} params.onHSClickHandler 触发衡石交互
 * @param {Array<clickEventData>} clickData 需要传递的数据
 */
